'use strict';

/* --------------------------------------------------------------
 extend_order_status_modal_entry_selection.js 2018-12-03
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2018 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Extends the order status modal with a "entry selection" widget.
 */
(function () {
	'use strict';

	/**
  * Updates the dropdown button text with the number of the selected items.
  *
  * @param {jQuery} $buttonDropdown
  */

	var updateButtonDropdownText = function updateButtonDropdownText($buttonDropdown) {
		var text = $buttonDropdown.find('input:checkbox:checked').length + ' ' + KlarnaHub.Config.lang.SELECTED_ENTRIES;
		$buttonDropdown.find('button:first').text(text);
	};

	/**
  * Returns the order line display name, displayed as a button dropdown menu item.
  *
  * @param {Object} orderLine Contains order line data.
  *
  * @return {String}
  */
	var getOrderLineDisplayName = function getOrderLineDisplayName(orderLine) {
		var displayName = KlarnaHub.Config.lang.UNMATCHED_PRODUCT;

		if (!KlarnaHub.Config.order) {
			return displayName;
		}

		if (orderLine.reference === 'ot_shipping' || orderLine.type === 'surcharge') {
			return orderLine.name.replace(/:$/, '');
		}

		// Product properties (format: "#x#" or "#{#}#").
		var reference = orderLine.merchant_data !== null ? orderLine.merchant_data : orderLine.reference;
		if (/\d+x\d+/.test(reference) || /\d+{\d+}\d+/.test(reference)) {
			var item = KlarnaHub.Config.order.items.find(function (item) {
				return item.addonValues.identifier === reference;
			});

			if (item) {
				displayName = item.attributes.map(function (attributes) {
					return attributes.name + ': ' + attributes.value;
				}).join(', ');
			}
		}

		// Product without properties or attributes.
		if (/^\d+$/.test(reference)) {
			displayName = '';
		}

		return orderLine.name + (displayName !== '' ? ' (' + displayName + ')' : '');
	};

	/**
  * Finds the capture the order line belongs to or returns null if the order line is not captured.
  *
  * @param {Object} orderLine Contains order line data.
  *
  * @return {Object|null}
  */
	var findCaptureWithOrderLine = function findCaptureWithOrderLine(orderLine) {
		return KlarnaHub.Config.klarnaOrder.captures.find(function (capture) {
			return capture.order_lines.find(function (capturedOrderLine) {
				return capturedOrderLine.merchant_data !== null ? capturedOrderLine.merchant_data === orderLine.merchant_data : capturedOrderLine.reference === orderLine.reference;
			});
		});
	};

	/**
  * Initializes the module.
  *
  * @private
  */
	var init = function init() {
		var klarnaOrder = KlarnaHub.Config.klarnaOrder;

		if (!klarnaOrder) {
			return;
		}

		// Filter captured order lines out.
		var orderLines = klarnaOrder.order_lines.filter(function (orderLine) {
			var capture = findCaptureWithOrderLine(orderLine);
			return orderLine.reference !== 'ot_coupon' && orderLine.reference !== 'ot_discount' && orderLine.reference !== 'ot_gv' && !capture;
		});

		var $buttonDropdown = $('<div/>', {
			'data-gx-widget': 'button_dropdown',
			'html': [$('<div/>', {
				'id': 'klarna-selected-entries',
				'data-use-button_dropdown': 'true',
				'css': {
					'margin-left': '0',
					'margin-bottom': '0'
				},
				'html': [$('<button/>', {
					'text': '0 ' + KlarnaHub.Config.lang.SELECTED_ENTRIES,
					'on': {
						'click': function click(event) {
							$(event.target).next('button').trigger('click');
						}
					}
				}), $('<ul/>', {
					'html': orderLines.map(function (orderLine) {
						return $('<li/>', {
							'html': [$('<span/>', {
								'html': [$('<input/>', {
									'type': 'checkbox',
									'data': orderLine,
									'css': {
										'height': 'auto',
										'margin-top': '5px'
									},
									'prop': {
										'checked': true
									},
									'on': {
										'click': function click(event) {
											event.stopPropagation();
											updateButtonDropdownText($buttonDropdown);
										}
									}
								}), $('<span/>', {
									'text': getOrderLineDisplayName(orderLine),
									'css': {
										'display': 'inline-block',
										'padding-left': '5px'
									},
									'on': {
										'click': function click(event) {
											event.stopPropagation();
											$(event.target).prev('input:checkbox').trigger('click');
										}
									}
								})]
							})]
						});
					})
				})]
			})]
		});

		updateButtonDropdownText($buttonDropdown);

		var $allEntriesCaptured = $('<span/>', {
			'text': KlarnaHub.Config.lang.ALL_ENTRIES_CAPTURED
		});

		var $form = $('#update_orders_status_form');
		var $select = $form.find('[name="gm_status"]');

		// Add the notify-klarna-hub checkbox (hidden by default).
		var $sourceControlGroup = $form.find('[name="gm_notify_klarna"]').closest('.control-group');
		var $controlGroup = $sourceControlGroup.clone(true);
		$controlGroup.addClass('hidden');

		var $label = $controlGroup.find('label');
		$label.text(KlarnaHub.Config.lang.KLARNA_CAPTURE);
		$controlGroup.find('.single-checkbox').remove();

		if (orderLines.length) {
			$controlGroup.append($buttonDropdown);
			gx.widgets.init($controlGroup);
		} else {
			$controlGroup.append($allEntriesCaptured);
		}

		$controlGroup.insertBefore($sourceControlGroup.next());

		// Bind status type change event, toggle visibility.
		$select.on('change', function (event) {
			if (event.target.value && event.target.value == KlarnaHub.Config.orderStatusShipped) {
				$controlGroup.removeClass('hidden');
			} else {
				$controlGroup.addClass('hidden');
			}
		});
	};

	KlarnaHub.on('module', function (event) {
		if (event.module !== 'extend_order_status_modal') {
			return;
		}
		init();
	});
})();
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
